/* graymilter - simple laterlist mail filter module
**
** Copyright  2004,2025 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <sysexits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <pthread.h>
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#include <libmilter/mfapi.h>

#include "version.h"
#include "iptab.h"


/* Defines. */

#define MIN_UPDATE_INTERVAL 10	/* min seconds between updates */
#define MIN_FILE_AGE 30		/* min age of changed file before reading */

#define DEFAULT_LATERTIME 1500		/* 25 minutes */
#define DEFAULT_ALLOWTIME 172800	/* two days */
#define MAX_LISTS 10
#define N_LISTS 10

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))


/* Forwards. */

static void usage( void );

static void init_uid( const char* user );
static void init_socket( char* sockarg );
static void term_socket( char* sockarg );
static void init_daemon( void );
static void init_pidfile( const char* pidfilename );
static void term_pidfile( const char* pidfilename );
static void init_iptabs( void );
static void term_iptabs( void );

static void add_localhosts( iptab list );
static bool stat_file( time_t current_time, time_t mtime, char* filename );
static time_t read_file( iptab list, char* filename );
static void trim( char* str );
static void update( void );

/* The milter callback routines.  Signatures for these are fixed
** by the milter API.
*/
static sfsistat gray_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr );
static sfsistat gray_envfrom( SMFICTX* ctx, char** fromargs );
static sfsistat gray_close( SMFICTX* ctx );


/* Globals. */

static char* argv0;
static int latertime, allowtime;
static char* initial_allowlist_filenames[MAX_LISTS];
static time_t initial_allowlist_mtimes[MAX_LISTS];
static iptab initial_allowlists[MAX_LISTS];
static int n_initial_allowlists;
static ipaddress* laterlists[N_LISTS];
static int size_laterlists[N_LISTS];
static int count_laterlists[N_LISTS];
static iptab allowlists[N_LISTS];
static int n_allowlists[N_LISTS];
static int update_interval, later_update_interval, allow_update_interval;
static time_t last_update, last_later_update, last_allow_update;
static pthread_mutex_t lock;

static struct smfiDesc smfilter =
    {
    "GRAY",		/* filter name */
    SMFI_VERSION,	/* version code -- do not change */
    0,			/* flags */
    gray_connect,	/* connection info filter */
    NULL,		/* SMTP HELO command filter */
    gray_envfrom,	/* envelope sender filter */
    NULL,		/* envelope recipient filter */
    NULL,		/* header filter */
    NULL,		/* end of header */
    NULL,		/* body block filter */
    NULL,		/* end of message */
    NULL,		/* message aborted */
    gray_close,		/* connection cleanup */
    NULL,		/* unrecognized / unimplemented command */
    NULL,		/* DATA command filter  */
    NULL		/* negotiation  */
    };


int
main( int argc, char** argv )
    {
    int argn;
    char* user;
    char* pidfilename;
    int nodaemon;
    char* sockarg;

    /* Figure out the program's name. */
    argv0 = strrchr( argv[0], '/' );
    if ( argv0 != (char*) 0 )
	++argv0;
    else
	argv0 = argv[0];

    openlog( argv0, LOG_PERROR, LOG_MAIL );

    /* Parse args. */
    latertime = DEFAULT_LATERTIME;
    allowtime = DEFAULT_ALLOWTIME;
    n_initial_allowlists = 1;	/* 0 is reserved for local addresses */
    user = (char*) 0;
    pidfilename = (char*) 0;
    nodaemon = 0;
    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	{
	if ( ( strncmp( argv[argn], "-latertime", strlen( argv[argn] ) ) == 0 ||
	       strncmp( argv[argn], "-graytime", strlen( argv[argn] ) ) == 0 ) && argn + 1 < argc )
	    {
	    ++argn;
	    latertime = atoi( argv[argn] );
	    }
	else if ( ( strncmp( argv[argn], "-allowtime", strlen( argv[argn] ) ) == 0 ||
	            strncmp( argv[argn], "-whitetime", strlen( argv[argn] ) ) == 0 ) && argn + 1 < argc )
	    {
	    ++argn;
	    allowtime = atoi( argv[argn] );
	    }
	else if ( ( strncmp( argv[argn], "-initialallowlist", max( strlen( argv[argn] ), 3 ) ) == 0 ||
	            strncmp( argv[argn], "-initialwhitelist", max( strlen( argv[argn] ), 3 ) ) == 0 ) && argn + 1 < argc )
	    {
	    if ( n_initial_allowlists == MAX_LISTS )
		{
		syslog( LOG_ERR, "too many initial allowlists" );
		exit( EX_USAGE );
		}
	    ++argn;
	    initial_allowlist_filenames[n_initial_allowlists++] = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-user", max( strlen( argv[argn] ), 3 ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    user = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-pidfile", strlen( argv[argn] ) ) == 0 && argn + 1 < argc )
	    {
	    ++argn;
	    pidfilename = argv[argn];
	    }
	else if ( strncmp( argv[argn], "-nodaemon", strlen( argv[argn] ) ) == 0 )
	    nodaemon = 1;
	else if ( strncmp( argv[argn], "-X", strlen( argv[argn] ) ) == 0 )
	    nodaemon = 1;
	else
	    usage();
	++argn;
	}
    if ( argn >= argc )
	usage();
    sockarg = argv[argn++];
    if ( argn != argc )
	usage();
    if ( latertime < 1 || allowtime < 1 || allowtime < latertime )
	usage();

    init_socket( sockarg );
    init_pidfile( pidfilename );
    init_uid( user );
    if ( ! nodaemon )
	init_daemon();
    init_iptabs();

    if ( pthread_mutex_init( &lock, (pthread_mutexattr_t*) 0 ) != 0 )
	{
	syslog( LOG_ERR, "pthread_mutex_init() failed" );
	exit( EX_OSERR );
	}

    later_update_interval = latertime / N_LISTS;
    allow_update_interval = allowtime / N_LISTS;
    if ( n_initial_allowlists > 1 )
	update_interval = MIN_UPDATE_INTERVAL;
    else
	update_interval = min( later_update_interval, allow_update_interval ) / 2;
    last_update = last_later_update = last_allow_update = time( (time_t*) 0 );

    syslog( LOG_NOTICE, "%s %s starting", GRAYMILTER_PROGRAM, GRAYMILTER_VERSION );
    if ( smfi_main() == MI_FAILURE )
	{
	syslog( LOG_ERR, "smfi_main() failed" );
	exit( EX_OSERR );
	}
    syslog( LOG_NOTICE, "%s %s terminating", GRAYMILTER_PROGRAM, GRAYMILTER_VERSION );

    (void) pthread_mutex_destroy( &lock );
    term_iptabs();
    term_pidfile( pidfilename );
    term_socket( sockarg );
    closelog();
    exit( EX_OK );
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage:  %s [-latertime/-graytime seconds] [-allowtime/-whitetime seconds] [-initialallowlist/-initialwhitelist file] [-user user] [-pidfile filename] [-nodaemon|-X] socket\n", argv0 );
    exit( EX_USAGE );
    }


static void
init_uid( const char* user )
    {
    struct passwd* pwd;
    char* ep;
    int uid;

    if ( getuid() == 0 )
	{
	/* If we're root, become another user. */
	if ( user == (char*) 0 )
	    syslog( LOG_WARNING, "warning: started as root but no -user flag specified" );
	else
	    {
	    /* Is it a number? */
	    uid = strtol( user, &ep, 0 );
	    if ( *ep == '\0' )
		pwd = getpwuid( uid );
	    else
		pwd = getpwnam( user );
	    if ( pwd == (struct passwd*) 0 )
		{
		syslog( LOG_ERR, "unknown user: '%s'", user );
		exit( EX_OSERR );
		}
	    /* Set aux groups to null. */
	    if ( setgroups( 0, (gid_t*) 0 ) < 0 )
		{
		syslog( LOG_ERR, "setgroups - %m" );
		exit( EX_OSERR );
		}
	    /* Set primary group. */
	    if ( setgid( pwd->pw_gid ) < 0 )
		{
		syslog( LOG_ERR, "setgid - %m" );
		exit( EX_OSERR );
		}

	    /* Try setting aux groups correctly - not critical if this fails. */
	    if ( initgroups( user, pwd->pw_gid ) < 0 )
		syslog( LOG_WARNING, "initgroups - %m" );
	    /* Set uid. */
	    if ( setuid( pwd->pw_uid ) < 0 )
		{
		syslog( LOG_ERR, "setuid - %m" );
		exit( EX_OSERR );
		}
	    }
	}
    else
	{
	/* If we're not root but got a -user flag anyway, that's an error. */
	if ( user != (char*) 0 )
	    {
	    syslog( LOG_ERR, "can't switch users if not started as root" );
	    exit( EX_USAGE );
	    }
	}
    }


static void
init_socket( char* sockarg )
    {
    /* Harden our umask so that the new socket gets created securely. */
    umask( 0077 );

    /* Initialize milter stuff. */
    if ( smfi_register( smfilter ) == MI_FAILURE )
	{
	syslog( LOG_ERR, "smfi_register() failed" );
	exit( EX_OSERR );
	}
    smfi_setconn( sockarg );
    if ( smfi_opensocket( true ) == MI_FAILURE )
	{
	syslog( LOG_ERR, "smfi_opensocket() failed" );
	exit( EX_OSERR );
	}
    }


static void
term_socket( char* sockarg )
    {
    }


static void
init_daemon( void )
    {
    /* Daemonize. */
#ifdef HAVE_DAEMON
    if ( daemon( 0, 0 ) < 0)
	{
	syslog( LOG_ERR, "daemon - %m" );
	exit( EX_OSERR );
	}
#else /* HAVE_DAEMON */
    switch ( fork() )
	{
	case 0:
	    break;  
	case -1:
	    syslog( LOG_ERR, "fork - %m" );
	    exit( EX_OSERR );
	default:
	    exit( EX_OK );
	}
#ifdef HAVE_SETSID
    setsid();
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON */
    }


static void
init_pidfile( const char* pidfilename )
    {
    if ( pidfilename != (char*) 0 )
	{
	FILE* fp;

	fp = fopen( pidfilename, "w" );
	if ( fp == (FILE*) 0 )
	    syslog( LOG_ERR, "unable to write PID file - %m" );
	else
	    {
	    (void) fprintf( fp, "%ld\n", (long) getpid() );
	    (void) fclose( fp );
	    }
	}
    }


static void
term_pidfile( const char* pidfilename )
    {
    if ( pidfilename != (char*) 0 )
	(void) unlink( pidfilename );
    }


static void
init_iptabs( void )
    {       
    int i;

    initial_allowlists[0] = iptab_new();
    add_localhosts( initial_allowlists[0] );
    for ( i = 1; i < n_initial_allowlists; ++i )
	{
	initial_allowlists[i] = iptab_new();
	if ( initial_allowlists[i] == (iptab) 0 )
	    {
	    syslog( LOG_ERR, "allowlist create failed" );
	    exit( EX_OSERR );
	    }
	initial_allowlist_mtimes[i] = read_file( initial_allowlists[i], initial_allowlist_filenames[i] );
	}

    for ( i = 0; i < N_LISTS; ++i )
	{
	size_laterlists[i] = 1000;	/* arbitrary */
	laterlists[i] = (ipaddress*) malloc( size_laterlists[i] * sizeof(ipaddress) );
	if ( laterlists[i] == (ipaddress*) 0 )
	    {
	    syslog( LOG_ERR, "laterlist create failed" );
	    exit( EX_OSERR );
	    }
	count_laterlists[i] = 0;
	allowlists[i] = iptab_new();
	if ( allowlists[i] == (iptab) 0 )
	    {
	    syslog( LOG_ERR, "allowlist create failed" );
	    exit( EX_OSERR );
	    }
	n_allowlists[i] = 0;
	}
    }


static void
term_iptabs( void )
    {
    int i;

    for ( i = 0; i < N_LISTS; ++i )
	{
	free( (void*) laterlists[i] );
	iptab_delete( allowlists[i] );
	}
    for ( i = 0; i < n_initial_allowlists; ++i )
	iptab_delete( initial_allowlists[i] );
    }


static void
add_localhosts( iptab list )
    {
    const ipaddress v4localhost = { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xff, 0xff, 127, 0, 0, 0 }, 104 };
    const ipaddress v6localhost = { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 }, 128 };

    (void) iptab_add( list, &v4localhost );
    (void) iptab_add( list, &v6localhost );
    }


/* Returns true if the file is still current, else false. */
static bool
stat_file( time_t current_time, time_t mtime, char* filename )
    {
    struct stat sb;

    if ( stat( filename, &sb ) < 0 )
	return true;	/* Can't stat it?  Ignore. */
    if ( sb.st_mtime == mtime )
	return true;	/* Unchanged. */
    if ( current_time - sb.st_mtime < MIN_FILE_AGE )
	return true;	/* Not old enough, we'll catch it next time. */
    return false;		/* Changed. */
    }


/* Reads one file into the database. */
static time_t
read_file( iptab list, char* filename )
    {
    FILE* fp;
    struct stat sb;
    time_t mtime;
    char line[1000];
    ipaddress ipa;

    syslog( LOG_INFO, "reading %s", filename );
    fp = fopen( filename, "r" );
    if ( fp == (FILE*) 0 )
	{
	syslog( LOG_ERR, "fopen '%s' - %m", filename );
	mtime = (time_t) -1;
	}
    else
	{
	if ( fstat( fileno(fp), &sb ) == 0 )
	    mtime = sb.st_mtime;
	else
	    mtime = (time_t) -1;
	while ( fgets( line, sizeof(line), fp ) != (char*) 0 )
	    {
	    trim( line );
	    if ( line[0] == '\0' )
		continue;
	    if ( iptab_parse_address( line, &ipa ) )
		(void) iptab_add( list, &ipa );
	    else
		syslog( LOG_INFO, "unparsable IP address - \"%s\"", line );
	    }
	(void) fclose( fp );
	}
    return mtime;
    }


static void
trim( char* str )
    {
    char* cp;
    int len;

    cp = strchr( str, '#' );
    if ( cp != (char*) 0 )
	*cp = '\0';
    len = strlen( str );
    while ( str[len-1] == '\n' || str[len-1] == '\r' || str[len-1] == ' ' || str[len-1] == '\t' )
	{
	--len;
	str[len] = '\0';
	}
    }


static void
update( void )
    {
    time_t current_time;
    int i;
    ipaddress* tmp_laterlist;
    int tmp_size_laterlist;
    iptab tmp_allowlist;

    current_time = time( (time_t*) 0 );
    if ( current_time - last_update < update_interval )
	return;
    last_update = current_time;

    if ( pthread_mutex_lock( &lock ) == 0 )
	{
	for ( i = 1; i < n_initial_allowlists; ++i )
	    {
	    if ( ! stat_file( current_time, initial_allowlist_mtimes[i], initial_allowlist_filenames[i] ) )
		{
		syslog( LOG_INFO, "initial allowlist file %d changed - autoupdating", i );
		iptab_clear( initial_allowlists[i] );
		initial_allowlist_mtimes[i] = read_file( initial_allowlists[i], initial_allowlist_filenames[i] );
		}
	    }

	while ( current_time - last_later_update >= later_update_interval )
	    {
	    /* Graduate addresses from the last laterlist to the first allowlist. */
	    syslog( LOG_INFO, "graduating %d addresses to allowlist", count_laterlists[N_LISTS-1] );
	    for ( i = 0; i < count_laterlists[N_LISTS-1]; ++i )
		(void) iptab_add( allowlists[0], &laterlists[N_LISTS-1][i] );
	    n_allowlists[0] += count_laterlists[N_LISTS-1];
	    /* Rotate the laterlists. */
	    tmp_laterlist = laterlists[N_LISTS-1];
	    tmp_size_laterlist = size_laterlists[N_LISTS-1];
	    for ( i = N_LISTS - 1; i > 0; --i )
		{
		laterlists[i] = laterlists[i-1];
		size_laterlists[i] = size_laterlists[i-1];
		count_laterlists[i] = count_laterlists[i-1];
		}
	    laterlists[0] = tmp_laterlist;
	    size_laterlists[0] = tmp_size_laterlist;
	    /* Clear out the first laterlist. */
	    count_laterlists[0] = 0;
	    /* Bump update time. */
	    last_later_update += later_update_interval;
	    }

	while ( current_time - last_allow_update >= allow_update_interval )
	    {
	    syslog( LOG_INFO, "expiring %d addresses from allowlist", n_allowlists[N_LISTS-1] );
	    /* Rotate the allowlists. */
	    tmp_allowlist = allowlists[N_LISTS-1];
	    for ( i = N_LISTS - 1; i > 0; --i )
		{
		allowlists[i] = allowlists[i-1];
		n_allowlists[i] = n_allowlists[i-1];
		}
	    allowlists[0] = tmp_allowlist;
	    /* Clear out the first allowlist. */
	    iptab_clear( allowlists[0] );
	    n_allowlists[0] = 0;
	    /* Bump update time. */
	    last_allow_update += allow_update_interval;
	    }

	(void) pthread_mutex_unlock( &lock );
	}
    }


/* The private data struct. */
struct connection_data {
    ipaddress ipa;
    int action;
    char *connhost;
    };
#define ACTION_UNKNOWN 0
#define ACTION_TEMPFAIL 1


/* gray_connect - handle the initial TCP connection
**
** Called at the start of a connection.  Any per-connection data should
** be initialized here.
**
** connhost: The hostname of the client, based on a reverse lookup.
** connaddr: The client's IP address, based on getpeername().
*/
static sfsistat
gray_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr )
    {
    struct connection_data* cd;
    int i;
    char str[100];

    update();

    if ( connaddr == (_SOCK_ADDR*) 0 )
	{
	syslog( LOG_INFO, "no sockaddr - accepting" );
	return SMFIS_ACCEPT;
	}

    cd = (struct connection_data*) malloc( sizeof(struct connection_data) );
    if ( cd == (struct connection_data*) 0 )
	{
	syslog( LOG_ERR, "couldn't allocate connection_data" );
	return SMFIS_ACCEPT;
	}
    (void) smfi_setpriv( ctx, (void*) cd );
    cd->action = ACTION_UNKNOWN;

    if ( connaddr->sa_family == AF_INET )
	{
	struct sockaddr_in* sa_in;
	unsigned char* uchar_addr;

	sa_in = (struct sockaddr_in*) ( (void*) connaddr );	/* extra cast to avoid alignment warning */
	uchar_addr = (unsigned char*) &sa_in->sin_addr.s_addr;
	cd->ipa.octets[0] = cd->ipa.octets[1] = cd->ipa.octets[2] = cd->ipa.octets[3] = cd->ipa.octets[4] = cd->ipa.octets[5] = cd->ipa.octets[6] = cd->ipa.octets[7] = cd->ipa.octets[8] = cd->ipa.octets[9] = 0;
	cd->ipa.octets[10] = cd->ipa.octets[11] = 0xff;
	cd->ipa.octets[12] = uchar_addr[0];
	cd->ipa.octets[13] = uchar_addr[1];
	cd->ipa.octets[14] = uchar_addr[2];
	cd->ipa.octets[15] = uchar_addr[3];
	cd->ipa.prefixlen = 128;
	}
    else if ( connaddr->sa_family == AF_INET6 )
	{
	struct sockaddr_in6* sa_in6;
	unsigned char* uchar_addr;

	sa_in6 = (struct sockaddr_in6*) ( (void*) connaddr );	/* extra cast to avoid alignment warning */
	uchar_addr = (unsigned char*) &sa_in6->sin6_addr.s6_addr;
	cd->ipa.octets[0] = uchar_addr[0];
	cd->ipa.octets[1] = uchar_addr[1];
	cd->ipa.octets[2] = uchar_addr[2];
	cd->ipa.octets[3] = uchar_addr[3];
	cd->ipa.octets[4] = uchar_addr[4];
	cd->ipa.octets[5] = uchar_addr[5];
	cd->ipa.octets[6] = uchar_addr[6];
	cd->ipa.octets[7] = uchar_addr[7];
	cd->ipa.octets[8] = uchar_addr[8];
	cd->ipa.octets[9] = uchar_addr[9];
	cd->ipa.octets[10] = uchar_addr[10];
	cd->ipa.octets[11] = uchar_addr[11];
	cd->ipa.octets[12] = uchar_addr[12];
	cd->ipa.octets[13] = uchar_addr[13];
	cd->ipa.octets[14] = uchar_addr[14];
	cd->ipa.octets[15] = uchar_addr[15];
	cd->ipa.prefixlen = 128;
	}
    else
	{
	syslog( LOG_INFO, "unknown address family - accepting" );
	return SMFIS_ACCEPT;
	}

    cd->connhost = strdup( connhost );	/* don't care about failure */

    /* Add the address to the first laterlist. */
    if ( count_laterlists[0] >= size_laterlists[0] )
	{
	/* Oosp, got to expand the list. */
	if ( pthread_mutex_lock( &lock ) != 0 )
	    {
	    syslog( LOG_ERR, "pthread_mutex_lock() failed" );
	    return SMFIS_ACCEPT;
	    }
	size_laterlists[0] *= 2;
	syslog( LOG_NOTICE, "expanding a laterlist to %d items", size_laterlists[0] );
	laterlists[0] = (ipaddress*) realloc( (void*) laterlists[0], size_laterlists[0] * sizeof(ipaddress) );
	(void) pthread_mutex_unlock( &lock );
	}
    laterlists[0][count_laterlists[0]] = cd->ipa;
    ++count_laterlists[0];

    /* Check if the address is in the initial allowlists. */
    for ( i = 0; i < n_initial_allowlists; ++i )
	if ( iptab_check( initial_allowlists[i], &cd->ipa ) )
	    {
	    syslog( LOG_INFO, "%s (%s) is allowlisted - accepting", iptab_format_address( &cd->ipa, str, sizeof(str) ), cd->connhost );
	    return SMFIS_ACCEPT;
	    }

    /* Check if the address is in any of the allowlists. */
    for ( i = 0; i < N_LISTS; ++i )
	{
	if ( iptab_check( allowlists[i], &cd->ipa ) )
	    {
	    syslog( LOG_INFO, "%s (%s) has been seen before - accepting", iptab_format_address( &cd->ipa, str, sizeof(str) ), cd->connhost );
	    return SMFIS_ACCEPT;
	    }
	}

    /* Tempfail it.  But we can't actually send the response yet, because
    ** we're not allowed to call smfi_setreply() from the _connect
    ** callback.
    */
    cd->action = ACTION_TEMPFAIL;
    return SMFIS_CONTINUE;
    }


/* gray_envfrom - handle the MAIL FROM:<> command
**
** Called at the start of each message.  There may be multiple messages
** in a connection.  Any per-message data should be initialized here.
**
** fromargs: Null-terminated SMTP command arguments; fromargs[0] is the
**                 sender address.
*/
static sfsistat
gray_envfrom( SMFICTX* ctx, char** fromargs )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );
    char* auth_type;
    char* verify;
    char str[100];

    /* The temporary failure response can't happen in the connect handler
    ** because we haven't started speaking SMTP protocol yet.  You need to
    ** speak SMTP in order to send back an SMTP error response.
    **
    ** It used to happen in the HELO handler, which is the next one to run.
    **
    ** Now it is here in the MAIL FROM handler, because this is the first
    ** place where we can check the authentication macros.
    */

    /* Authenticated messages get accepted without question. */
    auth_type = smfi_getsymval( ctx, "{auth_type}" );
    verify = smfi_getsymval( ctx, "{verify}" );
    if ( auth_type != (char*) 0 ||
         ( verify != (char*) 0 && strcmp( verify, "OK" ) == 0 ) )
	{
	syslog( LOG_INFO, "%s (%s) is authenticated - accepting", iptab_format_address( &cd->ipa, str, sizeof(str) ), cd->connhost );
	return SMFIS_ACCEPT;
	}

    if ( cd->action == ACTION_TEMPFAIL )
	{
	/* Send back the temporary failure code. */
#ifdef notdef
	syslog( LOG_INFO, "%d.%d.%d.%d (%s) is new - tempfailing", cd->ipa.a, cd->ipa.b, cd->ipa.c, cd->ipa.d, cd->connhost );
#endif
	(void) smfi_setreply( ctx, "421", "4.3.2", "laterlisted - please try again later" );
	return SMFIS_TEMPFAIL;
	}

    return SMFIS_CONTINUE;
    }


/* gray_close - handle the connection being closed
**
** Called once at the end of a connection.  Any per-connection data
** should be freed here.
*/
static sfsistat
gray_close( SMFICTX* ctx )
    {
    struct connection_data* cd = (struct connection_data*) smfi_getpriv( ctx );

    if ( cd != (struct connection_data*) 0 )
	{
	(void) smfi_setpriv( ctx, (void*) 0 );
	if ( cd->connhost != (char*) 0 )
	    free( cd->connhost );
	free( (void*) cd );
	}

    return SMFIS_CONTINUE;
    }
